/**********************************************************************************/
/* MyFirstDeviceDiver.c													*/
/*																		*/
/* Bruno van Dooren														*/
/* 																		*/
/* a project to learn device driver development. it simulates a volatile storage	*/
/* device. in other words you can read and write a number of registers. these		*/
/* registers are kept in RAM. in a real enviroment the registers could be located	*/
/* on a PCI card or so. for the purpose of learning device driver basics, a		*/
/* simulation will do.														*/
/* please note that exessive use of comments was applied for learning purposes	*/
/***********************************************************************************/

#define _BUILDING_MY_FIRST_DEVICE_DRIVER_

#include <KernelExport.h>
#include <Drivers.h>
#include <Errors.h>
#include <OS.h>
#include <string.h>
#include <SupportDefs.h>
#include "MyFirstDeviceDriver.h"

/*global to tell the kernel which platform version was used when building the driver*/
int32	api_version = B_CUR_DRIVER_API_VERSION;

/* null-terminated array of device names supported by this driver	*/
static const char *my_device_name[] = {
	"misc/MyFirstDeviceDriver",
	NULL
};

/*handle to the driver settings file*/
void* settings;

/*store the debugging flag*/
bool EnableDebug;

/*semaphore for synchronizing reading and writing to the memory*/
sem_id LocalSem;

/*array of integers for simulating a volatile register range. this range would normally
be implemented on a add-on card. access to the register range needs to be synched*/
#define REG_SIZE		1024
int RegRange[REG_SIZE];

/*this struct represents session specific information. each 'open' command will give
birth to a new SessionCookie. callers should not lose the cookie pointer, or they will
never find their cookie again :(.
note: smart schemes may be implemented to free every cookie when the driver is unloaded*/
struct SessionCookie
{
	int NumOps;	/*the number of operations that have been executed with this cookie*/
};

/* function pointers for the device hooks entry points	*/
device_hooks my_device_hooks = {
	my_device_open, 			/* -> open entry point */
	my_device_close, 			/* -> close entry point */
	my_device_free,			/* -> free cookie */
	my_device_control, 		/* -> control entry point */
	my_device_read,			/* -> read entry point */
	my_device_write			/* -> write entry point */
};

/**********************************************************************************/
/* init_hardware - called once the first time the driver is loaded.				*/
/* synchronization is not needed, because it is executed only once				*/
/**********************************************************************************/
status_t
init_hardware (void)
{
	/*open the settings file for this driver. it is located in ~/config/settings/kernel/drivers*/
	settings = load_driver_settings("MyFirstDeviceDriver");
	
	/*get the state of the debug flag. if the parameter was found, but no value defined, the return
	is 'true'. if the parameter was not found, false is returned.*/
	EnableDebug = get_driver_boolean_parameter(settings, "debug", false, true);
	
	/*close the settings file.*/
	unload_driver_settings(settings);
	
	/*set the debugging state*/
	set_dprintf_enabled(EnableDebug);
	
	dprintf("MFDD/init_hardware();\n");
	return B_OK;
	/*must return B_OK to tell the kernel that the hardware was detected.*/
	/*otherwise the (simulated) hardware will not be useable*/
}

/**********************************************************************************/
/*	init_driver - optional function - called every time the driver is loaded.		*/
/* synchronization is not an issue, since this function is executed only once.	*/
/**********************************************************************************/
status_t
init_driver (void)
{
	dprintf("MFDD/init_driver();\n");
	
	/*create a semaphore for synchronizing device access. initialize with 1. a 0
	doesn't work*/
	LocalSem = create_sem(1,"MyDeviceDriverSem");
	dprintf("MFDD/created sem with sem_id %d\n",LocalSem);
	
	/*transfer ownership of the sem to the kernel. this is necessarry because the thread
	trying to aqcuire the sem does not have to be the thread that created it.*/
	set_sem_owner(LocalSem, B_SYSTEM_TEAM);
	
	return B_OK;
}


/**********************************************************************************/
/* uninit_driver - optional function - called every time the driver is unloaded	*/
/* synchronization is not an issue, since this function is executed only once.	*/
/**********************************************************************************/
void
uninit_driver (void)
{
	dprintf("MFDD/uninit_driver();\n");
	/*in this function you can clean up all data that was created, initialized
	by init_driver*/
	/*delete the device access semaphore*/
	delete_sem(LocalSem);
	dprintf("MFDD/delete sem with sem_id %d \n",LocalSem);
}

	
/**********************************************************************************/
/* my_device_open - handle open() calls										*/
/* synchronization is important, because multiple 'open' operations can be		*/
/* executed at the same time.												*/
/**********************************************************************************/
static status_t
my_device_open (const char *name, uint32 flags, void** cookie)
{
	int i = 0;
	struct SessionCookie * NewCookie;
	dprintf("MFDD/my_device_open(%s,%d,%d);\n",name,flags,cookie);
	
	/*check if the supplied name is an existing device*/
	while(my_device_name[i] != NULL)
	{
		if (!strcmp(name, my_device_name[i]))
		{
			/*create a new cookie*/
			NewCookie = (struct SessionCookie*)malloc(sizeof(*NewCookie));
			if(!NewCookie)
			{
				/*no memory to create cookie*/
				dprintf("MFDD/Could not create cookie. No more room in the oven\n");
				return B_NO_MEMORY;
			}
			NewCookie->NumOps = 0;
			*cookie = NewCookie;
			dprintf("MFDD/Cookie address was %d\n",*cookie);
			return B_OK;
		}
		i++;
	}
		
	dprintf("MFDD/device was unknown. sorry: no cookie :)\n");
	return B_NAME_NOT_FOUND;
}


/**********************************************************************************/
/* my_device_read - handle read() calls										*/
/**********************************************************************************/
static status_t
my_device_read (void* cookie, off_t position, void *buf, size_t* num_bytes)
{
	int *IntArray = 0;
	int i = 0;
	int NumInts = 0;
	struct SessionCookie * MySessionCookie = cookie;
	status_t ErrorCode = B_NO_ERROR;
	
	/*print calling information. note that position is a 64 bit value: long long.
	if it is not cast to long, the dprintf function will read it as a 32 bit int.
	the values for buf and num_bytes will be reported with wrong values*/
	dprintf("MFDD/my_device_read(%d,%d,%d,%d);\n",
							cookie,
							(long)position,
							buf,
							num_bytes);
	
	/*read a number of ints from the buffer*/
	IntArray = buf;
	NumInts = *num_bytes / sizeof(int);

	//synchronize device access
	ErrorCode = acquire_sem(LocalSem);
	if(B_NO_ERROR == ErrorCode)
	{
		for(i=0; i<NumInts; i++)
			if((0 < i) || (REG_SIZE >= i))
			{
				IntArray[i] = RegRange[i + position];
				dprintf("MFDD/read pos %d -> value = %d\n",
							(long)(i + position),
							RegRange[i + position]);
			}
			else
				/*at the end of the write action, num_bytes should reflect the number
				of bytes the were actually written.*/
				*num_bytes -= sizeof(int8) / sizeof(int);
		MySessionCookie->NumOps++;
		release_sem(LocalSem);	//only need to aqcuire sem if it has been aquired
	}
	
	dprintf("MFDD/num_bytes was %d\n",*num_bytes);
	return ErrorCode;
}


/**********************************************************************************/
/* my_device_write - handle write() calls										*/
/**********************************************************************************/

static status_t
my_device_write (void* cookie, off_t position, const void* buffer, size_t* num_bytes)
{
	int *IntArray = 0;
	int i = 0;
	int NumInts = 0;
	struct SessionCookie * MySessionCookie = cookie;
	status_t ErrorCode = B_NO_ERROR;
	
	/*print calling information. note that position is a 64 bit value: long long.
	if it is not cast to long, the dprintf function will read it as a 32 bit int.
	the values for buf and num_bytes will be reported with wrong values*/
	dprintf("MFDD/my_device_write(cookie %d, position %d,buffer %d, buffersize%d);\n",
						cookie,
						(long)position,
						buffer,
						num_bytes);
	
	/*write a number of ints to the buffer*/
	IntArray = buffer;
	NumInts = *num_bytes / sizeof(int);
	
	//synchronize device access
	ErrorCode = acquire_sem(LocalSem);
	if(B_NO_ERROR == ErrorCode)
	{
		for(i=0; i<NumInts; i++)
			if((0 < i) || (REG_SIZE >= i))
			{
				RegRange[i + position] = IntArray[i];
				dprintf("MFDD/write value %d at position %d\n",
							RegRange[i + position],
							(long)(i + position)); 
			}
			else
				/*at the end of the write action, num_bytes should reflect the number
				of bytes the were actually written.*/
				*num_bytes -= sizeof(int);
		MySessionCookie->NumOps++;
		release_sem(LocalSem);		
	}

	dprintf("MFDD/num_bytes written: %d\n", *num_bytes);
	return ErrorCode;
}


/**********************************************************************************/
/* my_device_control - handle ioctl calls										*/
/**********************************************************************************/

static status_t
my_device_control (void* cookie, uint32 op, void* arg, size_t len)
{
	status_t ErrorCode = B_NO_ERROR;
	struct SessionCookie * MySessionCookie = cookie;
	
	dprintf("MFDD/my_device_control(%d,%d,%d,%d)\n;",cookie,op,arg,len);

	switch(op)
	{
	case op_NOP:		/*do nothing, as NOP specifies No OPeration*/
		break;
	case op_CLEARALL:	/*clear all registers*/
		dprintf("MFDD/operation op_CLEARALL\n");
		ErrorCode = acquire_sem(LocalSem);
		if(B_NO_ERROR == ErrorCode)
		{
			int i = 0;
			dprintf("MFDD/sem aquired\n");
			for(i=0; i<REG_SIZE; i++)
				RegRange[i] = 0;
			MySessionCookie->NumOps++;
			release_sem(LocalSem);
		}
		break;
	case op_GETNUMOPS:	/*get the number of operations that have been performed*/
		ErrorCode = acquire_sem(LocalSem);
		if(B_NO_ERROR == ErrorCode)
		{
			*((int*)arg) = MySessionCookie->NumOps;
			release_sem(LocalSem);
		}
		break;
	default:
		return B_DEV_INVALID_IOCTL;
	}
	return ErrorCode;
}


/**********************************************************************************/
/* my_device_close - handle close() calls										*/
/**********************************************************************************/
static status_t
my_device_close (void* cookie)
{
	dprintf("MFDD/my_device_close (%d)\n", cookie);
	/*contrary to what you might think, do not free the cookie here. there may
	still be operations in progress with the cookie in other threads. if other operations
	are in progress, freeing the cookie here could cause a system crash if it is used
	elsewhere after freeing it, and we wouldn't want to do that, would we?
	of course the close call should be made after all outstanding connections have been
	closed.*/
	return B_OK;
}


/**********************************************************************************/
/* my_device_free - called after the last device is closed, and after			*/
/* all i/o is complete.													*/
/**********************************************************************************/
static status_t
my_device_free (void* cookie)
{
	dprintf("MFDD/my_device_free (%d)\n", cookie);
	/*here you can free the cookie, because this function is only executed when all
	outstanding transactions have been finished.*/
	free(cookie);
	return B_OK;
}


/**********************************************************************************/
/* publish_devices - return a null-terminated array of devices					*/
/* supported by this driver.												*/
/**********************************************************************************/
const char**
publish_devices()
{
	dprintf("MFDD/publish_devices ()\n");
	/*return a null terminated array of c-strings. the kernel will publish these
	devices relative to the dev/ filesystem. for example: our device will be published
	as ~config/add-ons/kernel/drivers/dev/misc/MyFirstDeviceDriver*/
	return my_device_name;
}

/**********************************************************************************/
/* find_device - return ptr to device hooks structure for a						*/
/* given device name														*/
/**********************************************************************************/
device_hooks*
find_device(const char* name)
{
	int i = 0;
	dprintf("MFDD/find_device (%s)\n",name);
	/*check if the device is one of our published devices*/
	while(my_device_name[i] != NULL)
		if (!strcmp(name, my_device_name[i]))
			return &my_device_hooks;

	return NULL;
}

